<?php
declare (strict_types=1);

namespace app\api\controller\web;

use app\api\service\user\UserService;
use app\api\validate\web\UserValidate;
use app\dict\sys\AppTypeDict;
use core\base\BaseApiController;
use core\util\TokenAuth;

/**
 * 用户控制器
 *
 * @OA\Tag(
 *     name="用户",
 *     description="WEB端用户认证相关接口"
 * )
 */
class UserController extends BaseApiController
{
    /**
     * @OA\Post(
     *     path="/web/user/login",
     *     summary="用户登录",
     *     tags={"WEB"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"mobile", "type"},
     *             @OA\Property(property="mobile", type="string", description="手机号", example="13800138000"),
     *             @OA\Property(property="type", type="integer", description="登录方式：1=验证码 2=密码", example=1),
     *             @OA\Property(property="code", type="string", description="验证码（type=1时必填）", example="123456"),
     *             @OA\Property(property="scene", type="string", description="场景值（type=1时必填）", example="mobile_login"),
     *             @OA\Property(property="password", type="string", description="密码（type=2时必填）数字+字母，最少8位", example="12345678WYF")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="SUCCESS"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="mobile", type="string", example="13800138000"),
     *                 @OA\Property(property="nickname", type="string", example="用户8000"),
     *                 @OA\Property(property="avatar", type="string", example=""),
     *                 @OA\Property(property="token", type="string", example="eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...")
     *             )
     *         )
     *     )
     * )
     */
    public function login(): \think\Response
    {
        $params = (new UserValidate())->post()->goCheck('login');
        $userService = new UserService();

        if ($params['type'] == 1) {
            $scene = $params['scene'] ?? 'mobile_login';
            $result = $userService->loginByCode($params['mobile'], $params['code'] ?? '', $this->request->ip(), $scene);
        } else {
            $result = $userService->loginByPassword($params['mobile'], $params['password'] ?? '');
        }

        if (!$result) {
            return fail(UserService::getError());
        }

        return success('登录成功',$result);
    }

    /**
     * @OA\Post(
     *     path="/web/user/password/reset",
     *     summary="重置密码",
     *     tags={"WEB"},
     *     security={{"ApiKeyAuth": {}}},
     *          @OA\Parameter(
     *          name="Token",
     *          in="header",
     *          description="登录Token",
     *          required=true,
     *          @OA\Schema(type="string")
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"mobile", "code", "password"},
     *             @OA\Property(property="mobile", type="string", description="手机号", example="13800138000"),
     *             @OA\Property(property="code", type="string", description="验证码", example="123456"),
     *             @OA\Property(property="scene", type="string", description="场景值", example="reset_password"),
     *             @OA\Property(property="password", type="string", description="新密码", example="12345678WYF")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="密码重置成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function resetPassword(): \think\Response
    {
        $params = (new UserValidate())->post()->goCheck('reset');
        $scene = $params['scene'] ?? 'reset_password';
        $result = (new UserService())->resetPassword($params['mobile'], $params['code'], $params['password'], $scene);

        if (!$result) {
            return fail(UserService::getError());
        }

        return success('密码重置成功');
    }

    /**
     * @OA\Get(
     *     path="/web/user/info",
     *     summary="获取用户信息",
     *     tags={"WEB"},
     *     security={{"ApiKeyAuth": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=true,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="SUCCESS"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="nickname", type="string", example="用户8000"),
     *                 @OA\Property(property="avatar", type="string", example=""),
     *                 @OA\Property(property="user_no", type="string", example="OSC20231015001"),
     *                 @OA\Property(property="login_time", type="string", example="2023-10-15 12:00:00"),
     *                 @OA\Property(property="password_set", type="integer", description="登录密码设置状态 2=未设置 1=已设置", example=1)
     *             )
     *         )
     *     )
     * )
     */
    public function info(): \think\Response
    {
        $result = (new UserService())->getUserInfo($this->userId);

        if (!$result) {
            return fail(UserService::getError());
        }

        return success('请求成功',$result);
    }

    /**
     * @OA\Post(
     *     path="/web/user/profile",
     *     summary="修改个人信息",
     *     tags={"WEB"},
     *     security={{"ApiKeyAuth": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=true,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             @OA\Property(property="avatar", type="string", description="头像", example="https://example.com/storage/material/20260128/xxx.png"),
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="修改成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function profile(): \think\Response
    {
        $params = (new UserValidate())->post()->goCheck('profile');
        $result = (new UserService())->updateUserInfo($this->userId, $params);

        if (!$result) {
            return fail(UserService::getError());
        }

        return success('修改成功');
    }

    /**
     * @OA\Post(
     *     path="/web/user/logout",
     *     summary="退出登录",
     *     tags={"WEB"},
     *     security={{"ApiKeyAuth": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=true,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="退出成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function logout(): \think\Response
    {
        $token = $this->request->header('Token', '');
        TokenAuth::clearToken($this->userId, AppTypeDict::API, $token);
        return success('退出成功');
    }
}
