<?php
// +----------------------------------------------------------------------
// | 麦沃德科技赋能开发者，助力中小企业发展 
// +----------------------------------------------------------------------
// | Copyright (c) 2017～2024  www.wdadmin.cn    All rights reserved.
// +----------------------------------------------------------------------
// | Wdadmin系统产品软件并不是自由软件，不加密，并不代表开源，未经许可不可自由转售和商用
// +----------------------------------------------------------------------
// | Author: MY WORLD Team <bd@maiwd.cn>   www.wdadmin.cn
// +----------------------------------------------------------------------
namespace app\adminapi\validate\auth;

use app\adminapi\model\auth\Admin;
use core\base\BaseValidate;

class AdminValidate extends BaseValidate
{
    protected $rule = [
        'id' => 'require',
        'account' => 'require|length:1,32|unique:'.Admin::class,
        'name' => 'require|length:1,16|unique:'.Admin::class,
        'password' => 'require|length:6,32|edit',
        'password_confirm' => 'requireWith:password|confirm',
        'password_old' => 'requireWith:password',
        'role_id' => 'require',
        'disable' => 'require|in:0,1|checkDisable',
    ];

    protected $message = [
        'id.require' => '管理员id不能为空',
        'account.require' => '账号不能为空',
        'account.length' => '账号长度须在1-32位字符',
        'account.unique' => '账号已存在',
        'password.require' => '密码不能为空',
        'password.length' => '密码长度须在6-32位字符',
        'password_confirm.requireWith' => '确认密码不能为空',
        'password_confirm.confirm' => '两次输入的密码不一致',
        'password_old.requireWith' => '旧密码不能为空',
        'name.require' => '名称不能为空',
        'name.length' => '名称须在1-16位字符',
        'name.unique' => '名称已存在',
        'role_id.require' => '请选择角色',
        'disable.require' => '请选择状态',
        'disable.in' => '状态值错误',
        'avatar.require' => '请上传头像'
    ];

    public function sceneAdd()
    {
        return $this->only(['account', 'name', 'password', 'password_confirm', 'role_id', 'avatar', 'disable'])
            ->remove('password', 'edit')
            ->remove('disable', 'checkDisable');
    }

    public function sceneEdit()
    {
        return $this->remove('password', 'require|length')
            ->remove('role_id', 'require')
            ->remove('password_old', 'requireWith');
    }

    public function sceneProfile()
    {
        return $this->only(['account', 'name', 'password', 'password_confirm', 'password_old', 'avatar'])
            ->remove('account', 'unique')
            ->remove('name', 'unique') // 列表查询时name为可选，移除必填和唯一性验证
            ->append('avatar','require')
            ->remove('password', 'require|length');
    }
    
    public function sceneDisable()
    {
        return $this->only(['id', 'disable']);
    }



    /**
     * @Desc 禁用校验
     * @Create on 2025/12/9 下午3:10
     * @Create by woailuo
     */
    public function checkDisable($value, $rule, $data)
    {
        $admin = Admin::findOrEmpty($data['id']);
        if ($admin->isEmpty()) {
            return '管理员不存在';
        }

        if ($value && $admin['is_admin']) {
            return '超级管理员不允许被禁用';
        }
        return true;
    }

    public function sceneDetail()
    {
        return$this->only(['id']);
    }

    public function sceneDelete()
    {
        return$this->only(['id']);
    }

    public function sceneLists()
    {
        return $this->only(['name', 'account', 'page', 'limit'])
            ->remove('name', 'require|unique') // 列表查询时name为可选，移除必填和唯一性验证
            ->remove('account', 'require|unique'); // 列表查询时account为可选，移除必填和唯一性验证
    }

    /**
     * @notes 编辑情况下，检查是否填密码
     * @param $value
     * @param $rule
     * @param $data
     */
    public function edit($value, $rule, $data)
    {
        if (empty($data['password']) && empty($data['password_confirm'])) {
            return true;
        }
        $len = strlen($value);
        if ($len < 6 || $len > 32) {
            return '密码长度须在6-32位字符';
        }
        return true;
    }
}