<?php
declare (strict_types=1);

namespace app\adminapi\controller\message;

use app\adminapi\service\message\SmsTemplateService;
use app\adminapi\validate\message\SmsTemplateValidate;
use core\base\BaseAdminController;

/**
 * 短信模板控制器
 * Class SmsTemplateController
 * @package app\adminapi\controller\message
 *
 * @OA\Tag(
 *     name="短信模板",
 *     description="短信模板管理"
 * )
 */
class SmsTemplateController extends BaseAdminController
{
    /**
     * @Desc 短信模板列表
     * @Create on 2026/01/30
     *
     * @OA\Get(
     *     path="/message.smsTemplate/list",
     *     summary="短信模板列表",
     *     tags={"短信模板"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(name="page_no", in="query", description="页码", @OA\Schema(type="integer")),
     *     @OA\Parameter(name="page_size", in="query", description="每页数量", @OA\Schema(type="integer")),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="获取成功"),
     *             @OA\Property(property="data", type="object",
     *                  @OA\Property(property="list", type="array", @OA\Items(
     *                      @OA\Property(property="id", type="integer", description="ID"),
     *                      @OA\Property(property="template_name", type="string", description="模板名称"),
     *                      @OA\Property(property="template_key", type="string", description="模板Key"),
     *                      @OA\Property(property="template_value", type="string", description="模板值"),
     *                      @OA\Property(property="create_time", type="integer", description="创建时间"),
     *                  )),
     *                  @OA\Property(property="count", type="integer", description="总条数")
     *             )
     *         )
     *     )
     * )
     */
    public function list()
    {
        $params = $this->request->get();
        $service = new SmsTemplateService();
        $result = $service->getList($params);
        return success('获取成功', $result);
    }

    /**
     * @Desc 短信模板详情
     * @Create on 2026/01/30
     *
     * @OA\Get(
     *     path="/message.smsTemplate/detail",
     *     summary="短信模板详情",
     *     tags={"短信模板"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(name="id", in="query", description="模板ID", required=true, @OA\Schema(type="integer")),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="获取成功"),
     *             @OA\Property(property="data", type="object",
     *                  @OA\Property(property="id", type="integer", description="ID"),
     *                  @OA\Property(property="template_name", type="string", description="模板名称"),
     *                  @OA\Property(property="template_key", type="string", description="模板Key"),
     *                  @OA\Property(property="template_value", type="string", description="模板值"),
     *                  @OA\Property(property="create_time", type="integer", description="创建时间"),
     *             )
     *         )
     *     )
     * )
     */
    public function detail()
    {
        $id = (int)$this->request->get('id', 0);
        $service = new SmsTemplateService();
        $result = $service->getDetail($id);

        if (!$result) {
            return fail('模板不存在');
        }

        return success('获取成功', $result);
    }

    /**
     * @Desc 新增短信模板
     * @Create on 2026/01/30
     *
     * @OA\Post(
     *     path="/message.smsTemplate/create",
     *     summary="新增短信模板",
     *     tags={"短信模板"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"template_name", "template_key", "template_value"},
     *             @OA\Property(property="template_name", type="string", description="模板名称", example="登录验证码"),
     *             @OA\Property(property="template_key", type="string", description="模板Key", example="LOGIN_CODE"),
     *             @OA\Property(property="template_value", type="string", description="模板值/ID", example="SMS_123456789")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="创建成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function create()
    {
        $data = (new SmsTemplateValidate())->post()->goCheck('create');
        $service = new SmsTemplateService();
        $result = $service->create($data);

        if (!$result) {
            return fail($service->getError());
        }

        return success('创建成功');
    }

    /**
     * @Desc 更新短信模板
     * @Create on 2026/01/30
     *
     * @OA\Post(
     *     path="/message.smsTemplate/update",
     *     summary="更新短信模板",
     *     tags={"短信模板"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id", "template_name", "template_key", "template_value"},
     *             @OA\Property(property="id", type="integer", description="模板ID", example=1),
     *             @OA\Property(property="template_name", type="string", description="模板名称", example="登录验证码"),
     *             @OA\Property(property="template_key", type="string", description="模板Key", example="LOGIN_CODE"),
     *             @OA\Property(property="template_value", type="string", description="模板值/ID", example="SMS_123456789")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="更新成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function update()
    {
        $data = (new SmsTemplateValidate())->post()->goCheck('update');
        $service = new SmsTemplateService();
        $result = $service->update((int)$data['id'], $data);

        if (!$result) {
            return fail($service->getError());
        }

        return success('更新成功');
    }

    /**
     * @Desc 删除短信模板
     * @Create on 2026/01/30
     *
     * @OA\Post(
     *     path="/message.smsTemplate/delete",
     *     summary="删除短信模板",
     *     tags={"短信模板"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id"},
     *             @OA\Property(property="id", type="integer", description="模板ID", example=1)
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="删除成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function delete()
    {
        $id = (int)$this->request->post('id', 0);
        $service = new SmsTemplateService();
        $result = $service->delete($id);

        if (!$result) {
            return fail($service->getError());
        }

        return success('删除成功');
    }
}
