<?php
// +----------------------------------------------------------------------
// | 麦沃德科技赋能开发者，助力中小企业发展
// +----------------------------------------------------------------------
// | Copyright (c) 2017～2024  www.wdadmin.cn    All rights reserved.
// +----------------------------------------------------------------------
// | Wdadmin系统产品软件并不是自由软件，不加密，并不代表开源，未经许可不可自由转售和商用
// +----------------------------------------------------------------------
// | Author: MY WORLD Team <bd@maiwd.cn>   www.wdadmin.cn
// +----------------------------------------------------------------------

namespace app\adminapi\controller\material;

use app\common\service\FileService;
use core\base\BaseAdminController;
use app\adminapi\service\material\MaterialService;
use app\adminapi\validate\material\MaterialValidate;

/**
 * @OA\Tag(
 *     name="素材管理",
 *     description="素材中心-素材文件相关接口"
 * )
 */
class MaterialController extends BaseAdminController
{
    /**
     * @Desc 上传素材
     * @Create on 2025/12/19
     * @Create by system
     *
     * @OA\Post(
     *     path="/material.material/add",
     *     summary="上传素材",
     *     tags={"素材管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="multipart/form-data",
     *             @OA\Schema(
     *                 required={"file"},
     *                 @OA\Property(property="file", type="string", format="binary", description="上传文件"),
     *                 @OA\Property(property="category_id", type="integer", description="分类ID（可选）", example=0),
     *                 @OA\Property(property="type", type="string", description="素材类型（可选：image|video|file，不传则自动识别）", example="image")
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",     *
     *                 @OA\Property(property="key", type="string", example="material/20260128/xxx.png", description="相对路径"),
     *                 @OA\Property(property="url", type="string", example="/storage/material/20260128/xxx.png", description="访问链接"),
     *                 @OA\Property(property="type", type="string", example="image", description="文件类型"),
     *                 @OA\Property(property="size", type="integer", example=12345, description="文件大小"),
     *                 @OA\Property(property="name", type="string", example="原始文件名.png", description="原始文件名")
     *             )
     *         )
     *     )
     * )
     */
    public function add()
    {
        $params = (new MaterialValidate())->post()->goCheck('add');
        $saved = (new MaterialService())->add($params);

        return success('上传成功', [
            'key' => $saved['key'],
            'url' => FileService::getFileUrl(trim($saved['url'], '/')),
            'type' => $saved['type'],
            'size' => $saved['size'],
            'name' => $saved['name'],
        ]);        
    }

    /**
     * @Desc 素材列表（分页）
     * @Create on 2025/12/19
     * @Create by system
     *
     * @OA\Get(
     *     path="/material.material/lists",
     *     summary="素材列表（分页）",
     *     tags={"素材管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="type",
     *         in="query",
     *         description="素材类型：image|video|file（可选）",
     *         required=false,
     *         @OA\Schema(type="string", example="image")
     *     ),
     *     @OA\Parameter(
     *         name="category_id",
     *         in="query",
     *         description="分类ID（可选）",
     *         required=false,
     *         @OA\Schema(type="integer", example=0)
     *     ),
     *     @OA\Parameter(
     *         name="keyword",
     *         in="query",
     *         description="关键词（可选，按名称模糊）",
     *         required=false,
     *         @OA\Schema(type="string", example="")
     *     ),
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="页码",
     *         required=false,
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="limit",
     *         in="query",
     *         description="每页数量",
     *         required=false,
     *         @OA\Schema(type="integer", example=20)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(
     *                     property="list",
     *                     type="array",
     *                     @OA\Items(
     *                         @OA\Property(property="id", type="integer", description="素材ID"),
     *                         @OA\Property(property="category_id", type="integer", description="分类ID"),
     *                         @OA\Property(property="type", type="string", description="类型"),
     *                         @OA\Property(property="url", type="string", description="访问URL"),
     *                         @OA\Property(property="name", type="string", description="名称"),
     *                         @OA\Property(property="create_time", type="string", description="创建时间")
     *                     )
     *                 ),
     *                 @OA\Property(property="count", type="integer", example=0)
     *             )
     *         )
     *     )
     * )
     */
    public function lists()
    {
        $params = (new MaterialValidate())->get()->goCheck('lists');
        $data = (new MaterialService())->lists($params);
        return success('请求成功', $data);
    }

    /**
     * @Desc 素材详情
     * @Create on 2025/12/19
     * @Create by system
     *
     * @OA\Get(
     *     path="/material.material/detail",
     *     summary="素材详情",
     *     tags={"素材管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="id",
     *         in="query",
     *         description="素材ID",
     *         required=true,
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", description="素材ID"),
     *                 @OA\Property(property="category_id", type="integer", description="分类ID"),
     *                 @OA\Property(property="type", type="string", description="类型"),
     *                 @OA\Property(property="url", type="string", description="地址"),
     *                 @OA\Property(property="name", type="string", description="名称"),
     *                 @OA\Property(property="create_time", type="string", description="创建时间")
     *             )
     *         )
     *     )
     * )
     */
    public function detail()
    {
        $params = (new MaterialValidate())->get()->goCheck('detail');
        $data = (new MaterialService())->detail((int)$params['id']);
        return success('请求成功', $data);
    }

    /**
     * @Desc 删除素材（支持单条和批量）
     * @Create on 2025/12/19
     * @Create by system
     *
     * @OA\Post(
     *     path="/material.material/delete",
     *     summary="删除素材（支持单条和批量）",
     *     tags={"素材管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id"},
     *             @OA\Property(
     *                 property="id", 
     *                 description="素材ID（单条删除传整数如：1，批量删除传数组如：[1,2,3]）", 
     *                 oneOf={
     *                     @OA\Schema(type="integer", example=1),
     *                     @OA\Schema(type="array", @OA\Items(type="integer"), example={1, 2, 3})
     *                 }
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="删除成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function delete()
    {
        $params = (new MaterialValidate())->post()->goCheck('delete');
        (new MaterialService())->delete($params['id']);
        return success('删除成功');
    }

    /**
     * @Desc 批量移动分类
     * @Create on 2025/12/19
     * @Create by system
     *
     * @OA\Post(
     *     path="/material.material/move",
     *     summary="批量移动分类",
     *     tags={"素材管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id","category_id"},
     *             @OA\Property(
     *                 property="id",
     *                 type="array",
     *                 description="素材ID数组",
     *                 @OA\Items(type="integer", example=1)
     *             ),
     *             @OA\Property(property="category_id", type="integer", description="目标分类ID", example=1)
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", example=1),
     *             @OA\Property(property="msg", type="string", example="移动成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function move()
    {
        $params = (new MaterialValidate())->post()->goCheck('move');
        (new MaterialService())->move($params['id'], (int)$params['category_id']);
        return success('移动成功');
    }
}
