<?php
// +----------------------------------------------------------------------
// | 麦沃德科技赋能开发者，助力中小企业发展 
// +----------------------------------------------------------------------
// | Copyright (c) 2017～2024  www.wdadmin.cn    All rights reserved.
// +----------------------------------------------------------------------
// | Wdadmin系统产品软件并不是自由软件，不加密，并不代表开源，未经许可不可自由转售和商用
// +----------------------------------------------------------------------
// | Author: MY WORLD Team <bd@maiwd.cn>   www.wdadmin.cn
// +----------------------------------------------------------------------
namespace app\adminapi\controller\login;

use app\adminapi\service\login\LoginService;
use app\adminapi\validate\login\LoginValidate;
use core\base\BaseAdminController;

/**
 * 登录控制器
 * Class Login
 * @package app\adminapi\controller\login
 *
 * @OA\Tag(
 *     name="登录模块",
 *     description="后台登录相关接口"
 * )
 */
class LoginController extends BaseAdminController
{
    /**
     * @Desc 登录
     *
     * @OA\Post(
     *     path="/login.login/login",
     *     summary="管理员登录",
     *     tags={"登录模块"},
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"account", "password"},
     *             @OA\Property(property="account", type="string", description="账号", example="admin"),
     *             @OA\Property(property="password", type="string", description="密码", example="123456"),
     *             @OA\Property(property="captcha", type="string", description="验证码（网站设置开启验证码时必填）", example="ab12"),
     *             @OA\Property(property="captcha_key", type="string", description="验证码唯一标识", example="abc123...")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="SUCCESS"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="name", type="string", description="管理员名称", example="张三"),
     *                 @OA\Property(property="avatar", type="string", description="头像", example="http://framework.top/static/admin/image/default_avatar.png"),
     *                 @OA\Property(property="role_name", type="string", description="角色名称", example="员工/大厅经理"),
     *                 @OA\Property(property="token", type="string", description="登录Token", example="eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9...")
     *             )
     *         )
     *     )
     * )
     */
    public function login(): \think\Response
    {
        $params = (new LoginValidate())->post()->goCheck('login');
        // 获取验证码参数，传递给 LoginService
        $captcha = $params['captcha'] ?? '';
        $captchaKey = $params['captcha_key'] ?? '';
        $result = (new LoginService())->login($params['account'], $params['password'], $captcha, $captchaKey);
        if (!$result) {
            // 账号密码错误或验证码错误
            return fail(LoginService::getError());
        }
        return success($result);
    }

    /**
     * @Desc 退出登录
     *
     * @OA\Post(
     *     path="/login.login/logout",
     *     summary="退出登录",
     *     tags={"登录模块"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="退出成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function logout(): \think\Response
    {
        $result = (new LoginService())->logout($this->adminId);
        if (true === $result) {
            return success('退出成功');
        }
        return fail(LoginService::getError());
    }

}