<?php
// +----------------------------------------------------------------------
// | 麦沃德科技赋能开发者，助力中小企业发展
// +----------------------------------------------------------------------
// | Copyright (c) 2017～2024  www.wdadmin.cn    All rights reserved.
// +----------------------------------------------------------------------
// | Wdadmin系统产品软件并不是自由软件，不加密，并不代表开源，未经许可不可自由转售和商用
// +----------------------------------------------------------------------
// | Author: MY WORLD Team <bd@maiwd.cn>   www.wdadmin.cn
// +----------------------------------------------------------------------

namespace app\adminapi\controller\auth;

use core\base\BaseAdminController;
use app\adminapi\{
    service\auth\RoleService,
    validate\auth\RoleValidate,
};

/**
 * 角色控制器
 * Class RoleController
 * @package app\adminapi\controller\auth
 *
 * @OA\Tag(
 *     name="角色管理",
 *     description="角色相关接口"
 * )
 */
class RoleController extends BaseAdminController
{


    /**
     * @Desc 添加权限
     * @Create on 2025/12/4 上午9:51
     * @Create by woailuo
     *
     * @OA\Post(
     *     path="/auth.role/add",
     *     summary="添加角色",
     *     tags={"角色管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name"},
     *             @OA\Property(property="name", type="string", description="角色名称", maxLength=64, example="管理员"),
     *             @OA\Property(property="desc", type="string", description="备注", example="管理员角色描述"),
     *             @OA\Property(property="sort", type="integer", description="排序", example=0),
     *             @OA\Property(
     *                 property="menu_id",
     *                 type="array",
     *                 description="菜单ID数组，用于为角色分配权限",
     *                 @OA\Items(type="integer", description="菜单ID"),
     *                 example={1, 2, 3}
     *              )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="添加成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function add()
    {
        $params = (new RoleValidate())->post()->goCheck('add');
        (new RoleService)->add($params);
        return success('添加成功');
    }

    /**
     * @Desc 编辑角色
     * @Create on 2025/12/4
     *
     * @OA\Post(
     *     path="/auth.role/edit",
     *     summary="编辑角色",
     *     tags={"角色管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id", "name"},
     *             @OA\Property(property="id", type="integer", description="角色ID", example=1),
     *             @OA\Property(property="name", type="string", description="角色名称", maxLength=64, example="超级管理员"),
     *             @OA\Property(property="desc", type="string", description="备注", example="超级管理员角色描述"),
     *             @OA\Property(property="sort", type="integer", description="排序", example=0),
     *             @OA\Property(
     *                 property="menu_id",
     *                 type="array",
     *                 description="菜单ID数组，用于为角色分配权限",
     *                 @OA\Items(type="integer", description="菜单ID"),
     *                 example={1, 2, 3}
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="编辑成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function edit()
    {
        $params = (new RoleValidate())->post()->goCheck('edit');
        (new RoleService)->edit($params);
        return success('编辑成功');
    }

    /**
     * @Desc 删除角色
     * @Create on 2025/12/5 下午2:55
     * @Create by woailuo
     *
     * @OA\Post(
     *     path="/auth.role/delete",
     *     summary="删除角色(支持批量)",
     *     tags={"角色管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id"},
     *             @OA\Property(
     *                 property="id", 
     *                 description="角色ID (单条: 6, 多条: [8, 7, 6])",
     *                 oneOf={
     *                     @OA\Schema(type="integer", example=1),
     *                     @OA\Schema(type="array", @OA\Items(type="integer"), example={1, 2, 3})
     *                 }
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="删除成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function delete()
    {
        $params = (new RoleValidate())->post()->goCheck('delete');
        (new RoleService)->delete($params['id']);
        return success('删除成功');
    }

    /**
     * @Desc 角色列表
     * @Create on 2025/12/6 下午3:19
     * @Create by woailuo
     *
     * @OA\Get(
     *     path="/auth.role/lists",
     *     summary="角色列表",
     *     tags={"角色管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="name",
     *         in="query",
     *         description="角色名称（模糊搜索）",
     *         required=false,
     *         @OA\Schema(type="string", example="管理员")
     *     ),
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="页码",
     *         required=false,
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="limit",
     *         in="query",
     *         description="每页数量",
     *         required=false,
     *         @OA\Schema(type="integer", example=10)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="total", type="integer", description="总数", example=100),
     *                 @OA\Property(property="per_page", type="integer", description="每页数量", example=10),
     *                 @OA\Property(property="current_page", type="integer", description="当前页", example=1),
     *                 @OA\Property(
     *                     property="data",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="id", type="integer", description="角色ID", example=1),
     *                         @OA\Property(property="name", type="string", description="角色名称", example="管理员"),
     *                         @OA\Property(property="desc", type="string", description="备注", example="管理员角色"),
     *                         @OA\Property(property="sort", type="integer", description="排序", example=0),
     *                         @OA\Property(property="create_time", type="string", description="创建时间", example="2025-12-06 15:19:00"),
     *                         @OA\Property(
     *                             property="menu_id",
     *                             type="array",
     *                             description="角色拥有的菜单权限ID数组",
     *                             @OA\Items(type="integer"),
     *                             example={1, 2, 3}
     *                         )
     *                     )
     *                 )
     *             )
     *         )
     *     )
     * )
     */
    public function lists()
    {
        $data = (new RoleValidate())->get()->goCheck('lists');

        return success((new RoleService())->getPage($data));
    }

    /**
     * @Desc 添加管理员获取所有角色ID和名称
     * Create on 2025/12/24 上午9:32
     * Create by woailuo
     *
     * @OA\Get(
     *     path="/auth.role/getAllRoleIdAndName",
     *     summary="获取全部角色ID和名称",
     *     tags={"角色管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="array",
     *                 @OA\Items(
     *                     type="object",
     *                     @OA\Property(property="id", type="integer", description="角色ID", example=1),
     *                     @OA\Property(property="name", type="string", description="角色名称", example="管理员")
     *                 )
     *             )
     *         )
     *     )
     * )
     */
    public function getAllRoleIdAndName(){
        return success((new RoleService())->getAllRoleIdAndName());
    }
}