<?php
// +----------------------------------------------------------------------
// | 麦沃德科技赋能开发者，助力中小企业发展 
// +----------------------------------------------------------------------
// | Copyright (c) 2017～2024  www.wdadmin.cn    All rights reserved.
// +----------------------------------------------------------------------
// | Wdadmin系统产品软件并不是自由软件，不加密，并不代表开源，未经许可不可自由转售和商用
// +----------------------------------------------------------------------
// | Author: MY WORLD Team <bd@maiwd.cn>   www.wdadmin.cn
// +----------------------------------------------------------------------
namespace app\adminapi\controller\auth;

use app\adminapi\service\auth\AdminService;
use app\adminapi\validate\auth\AdminValidate;
use core\base\BaseAdminController;

/**
 * 管理员控制器
 * Class AdminController
 * @package app\adminapi\controller\auth
 *
 * @OA\Tag(
 *     name="管理员管理",
 *     description="管理员相关接口"
 * )
 */
class AdminController extends BaseAdminController
{
    /**
     * @Desc 添加管理员
     * @Create on 2025/12/9
     *
     * @OA\Post(
     *     path="/auth.admin/add",
     *     summary="添加管理员",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"name", "account", "password", "password_confirm", "disable", "role_id"},
     *             @OA\Property(property="name", type="string", description="管理员名称", maxLength=16, example="张三"),
     *             @OA\Property(property="account", type="string", description="账号", maxLength=32, example="admin001"),
     *             @OA\Property(property="password", type="string", description="密码", minLength=6, maxLength=32, example="123456"),
     *             @OA\Property(property="password_confirm", type="string", description="确认密码", minLength=6, maxLength=32, example="123456"),
     *             @OA\Property(property="avatar", type="string", description="头像URL", example=""),
     *             @OA\Property(property="disable", type="integer", description="是否禁用：0-否；1-是", enum={0, 1}, example=0),
     *             @OA\Property(
     *                 property="role_id",
     *                 type="array",
     *                 description="角色ID数组",
     *                 @OA\Items(type="integer"),
     *                 example={2, 3}
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="添加成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function add()
    {
        $params = (new AdminValidate())->post()->goCheck('add');
        $result = (new AdminService())->add($params);
        if (true === $result) {
            return success('添加成功');
        }
        return fail(AdminService::getError());
    }

    /**
     * @Desc 编辑管理员
     * @Create on 2025/12/9
     *
     * @OA\Post(
     *     path="/auth.admin/edit",
     *     summary="编辑管理员",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id", "name", "account"},
     *             @OA\Property(property="id", type="integer", description="管理员ID", example=1),
     *             @OA\Property(property="name", type="string", description="管理员名称", maxLength=16, example="李四"),
     *             @OA\Property(property="account", type="string", description="账号", maxLength=32, example="admin002"),
     *             @OA\Property(property="password", type="string", description="密码（可选）", minLength=6, maxLength=32, example=""),
     *             @OA\Property(property="password_confirm", type="string", description="确认密码（可选）", minLength=6, maxLength=32, example=""),
     *             @OA\Property(property="avatar", type="string", description="头像URL", example=""),
     *             @OA\Property(property="disable", type="integer", description="是否禁用：0-否；1-是", enum={0, 1}, example=0),
     *             @OA\Property(
     *                 property="role_id",
     *                 type="array",
     *                 description="角色ID数组（可选）",
     *                 @OA\Items(type="integer"),
     *                 example={1}
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="修改成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function edit()
    {
        $params = (new AdminValidate())->post()->goCheck('edit');
        $result = (new AdminService())->edit($params);
        if (true === $result) {
            return success('修改成功');
        }
        return fail(AdminService::getError());
    }

    /**
     * @Desc 更新个人信息
     * @Create on 2025/12/20
     *
     * @OA\Post(
     *     path="/auth.admin/profile",
     *     summary="更新个人信息",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id", "name", "account", "avatar"},
     *             @OA\Property(property="name", type="string", description="管理员名称", maxLength=16, example="李四"),
     *             @OA\Property(property="account", type="string", description="账号", maxLength=32, example="admin002"),
     *             @OA\Property(property="password", type="string", description="密码（可选）", minLength=6, maxLength=32, example=""),
     *             @OA\Property(property="password_confirm", type="string", description="确认密码（可选）", minLength=6, maxLength=32, example=""),
     *             @OA\Property(property="password_old", type="string", description="旧密码（可选）", minLength=6, maxLength=32, example=""),
     *             @OA\Property(property="avatar", type="string", description="头像URL", example="")
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="修改成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function profile()
    {
        $params = (new AdminValidate())->post()->goCheck('profile');
        $result = (new AdminService())->profile($params);
        if (true === $result) {
            return success('修改成功');
        }
        return fail(AdminService::getError());
    }

    /**
     * @Desc 查看管理员详情
     * @Create on 2025/12/9
     *
     * @OA\Get(
     *     path="/auth.admin/detail",
     *     summary="查看管理员详情",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="id",
     *         in="query",
     *         description="管理员ID",
     *         required=true,
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="获取成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="id", type="integer", description="管理员ID", example=1),
     *                 @OA\Property(property="name", type="string", description="管理员名称", example="张三"),
     *                 @OA\Property(property="account", type="string", description="账号", example="admin001"),
     *                 @OA\Property(property="avatar", type="string", description="头像URL", example=""),
     *                 @OA\Property(property="disable", type="integer", description="是否禁用：0-否；1-是", example=0),
     *                 @OA\Property(
     *                     property="role_id",
     *                     type="array",
     *                     description="角色ID数组",
     *                     @OA\Items(type="integer"),
     *                     example={1, 2}
     *                 ),
     *                 @OA\Property(property="role_name", type="string", description="角色名称（多个角色用/分隔）", example="管理员/经理"),
     *                 @OA\Property(property="is_admin", type="integer", description="是否是超级管理员：0-否；1-是", example=0),
     *             )
     *         )
     *     )
     * )
     */
    public function detail(){
        $params = (new AdminValidate())->get()->goCheck('detail');
        $detail = (new AdminService())->detail($params['id']);
        return success('请求成功',$detail);
    }

    /**
     * @Desc 删除管理员
     * @Create on 2025/12/9
     *
     * @OA\Post(
     *     path="/auth.admin/delete",
     *     summary="删除管理员",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id"},
     *             @OA\Property(
     *                 property="id", 
     *                 description="管理员ID（单条：6，批量：[6,7,8]）", 
     *                 oneOf={
     *                     @OA\Schema(type="integer", example=6),
     *                     @OA\Schema(type="array", @OA\Items(type="integer"), example={6, 7, 8})
     *                 }
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="删除成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function delete()
    {
        $params = (new AdminValidate())->post()->goCheck('delete');
        $result = (new AdminService())->delete($params['id']);
        if (true === $result) {
            return success('删除成功');
        }
        return fail(AdminService::getError());
    }

    /**
     * @Desc 修改管理员状态
     * @Create on 2025/12/26
     * 
     * @OA\Post(
     *     path="/auth.admin/disable",
     *     summary="修改管理员状态",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\JsonContent(
     *             required={"id", "disable"},
     *             @OA\Property(property="id", type="integer", description="管理员ID", example=1),
     *             @OA\Property(property="disable", type="integer", description="是否禁用：0-否；1-是", enum={0, 1}, example=1)
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="操作成功"),
     *             @OA\Property(property="data", type="object")
     *         )
     *     )
     * )
     */
    public function disable()
    {
        $params = (new AdminValidate())->post()->goCheck('disable');
        $result = (new AdminService())->disable($params);
        if (true === $result) {
            return success('操作成功');
        }
        return fail(AdminService::getError());
    }



    /**
     * @Desc 管理员列表
     * @Create on 2025/12/9
     *
     * @OA\Get(
     *     path="/auth.admin/lists",
     *     summary="管理员列表",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Parameter(
     *         name="name",
     *         in="query",
     *         description="管理员名称（模糊搜索）",
     *         required=false,
     *         @OA\Schema(type="string", example="张三")
     *     ),
     *     @OA\Parameter(
     *         name="account",
     *         in="query",
     *         description="账号（模糊搜索）",
     *         required=false,
     *         @OA\Schema(type="string", example="admin001")
     *     ),
     *     @OA\Parameter(
     *         name="page",
     *         in="query",
     *         description="页码，用于分页查询",
     *         required=false,
     *         @OA\Schema(type="integer", example=1)
     *     ),
     *     @OA\Parameter(
     *         name="limit",
     *         in="query",
     *         description="每页数量，用于分页查询",
     *         required=false,
     *         @OA\Schema(type="integer", example=10)
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(property="total", type="integer", description="总数", example=100),
     *                 @OA\Property(property="per_page", type="integer", description="每页数量", example=10),
     *                 @OA\Property(property="current_page", type="integer", description="当前页", example=1),
     *                 @OA\Property(
     *                     property="data",
     *                     type="array",
     *                     @OA\Items(
     *                         type="object",
     *                         @OA\Property(property="id", type="integer", description="管理员ID", example=1),
     *                         @OA\Property(property="name", type="string", description="管理员名称", example="张三"),
     *                         @OA\Property(property="account", type="string", description="账号", example="admin001"),
     *                         @OA\Property(property="avatar", type="string", description="头像URL", example=""),
     *                         @OA\Property(property="disable", type="integer", description="是否禁用：0-否；1-是", example=0),
     *                         @OA\Property(property="is_admin", type="integer", description="是否是超级管理员：0-否；1-是", example=0),
     *                         @OA\Property(property="create_time", type="integer", description="创建时间戳", example=1702123200),
     *                         @OA\Property(property="login_time", type="integer", description="最后登录时间戳", example=1702209600),
     *                         @OA\Property(property="login_ip", type="string", description="最后登录IP", example="192.168.1.1"),
     *                         @OA\Property(
     *                             property="role_id",
     *                             type="array",
     *                             description="管理员拥有的角色权限ID数组",
     *                             @OA\Items(type="integer"),
     *                             example={1, 2, 3}
     *                         ),
     *                         @OA\Property(property="role_name", type="string", description="角色名称（多个角色用/分隔）", example="管理员/经理")
     *                     )
     *                 )
     *             )
     *         )
     *     )
     * )
     */
    public function lists(): \think\Response
    {
        $data = (new AdminValidate())->get()->goCheck('lists');

        return success((new AdminService())->getPage($data));
    }

    /**
     * @Desc 获取管理员信息
     * @Create on 2025/12/16 上午10:42
     * @Create by woailuo
     *
     * @OA\Get(
     *     path="/auth.admin/getInfo",
     *     summary="获取管理员信息",
     *     tags={"管理员管理"},
     *     security={{"token": {}}},
     *     @OA\Parameter(
     *         name="Token",
     *         in="header",
     *         description="登录Token",
     *         required=false,
     *         @OA\Schema(type="string")
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="成功响应",
     *         @OA\JsonContent(
     *             @OA\Property(property="code", type="integer", description="操作状态：0=失败 1=成功", example=1),
     *             @OA\Property(property="msg", type="string", example="请求成功"),
     *             @OA\Property(
     *                 property="data",
     *                 type="object",
     *                 @OA\Property(
     *                     property="admin",
     *                     type="object",
     *                     description="管理员信息",
     *                     @OA\Property(property="id", type="integer", description="管理员ID", example=1),
     *                     @OA\Property(property="name", type="string", description="管理员名称", example="张三"),
     *                     @OA\Property(property="account", type="string", description="账号", example="admin001"),
     *                     @OA\Property(property="avatar", type="string", description="头像URL", example=""),
     *                     @OA\Property(
     *                          property="role_id",
     *                          type="array",
     *                          description="角色ID数组",
     *                          @OA\Items(type="integer"),
     *                          example={1}
     *                      ),
     *                     @OA\Property(
     *                         property="role_name",
     *                         type="array",
     *                         description="角色名称数组",
     *                         @OA\Items(type="string"),
     *                         example={"管理员", "经理"}
     *                     )
     *                 ),
     *                 @OA\Property(
     *                      property="menu",
     *                      type="array",
     *                      description="菜单列表",
     *                      @OA\Items(type="object")
     *                 ),
     *                 @OA\Property(
     *                      property="permissions",
     *                      type="array",
     *                      description="按钮权限标识列表",
     *                      @OA\Items(type="string"),
     *                      example={"sys:user:add", "sys:user:edit"}
     *                 )
     *             )
     *         )
     *     )
     * )
     */
    public function getInfo(): \think\Response
    {
        $adminInfo = (new AdminService())->getAdminInfo($this->adminId);
        return success('请求成功', $adminInfo);
    }
}